import React, { Component } from 'react';
import clone from 'clone';
import { ipcRenderer } from 'electron';
import { Switch, Route } from 'react-router';
import { withStyles } from '@material-ui/core/styles';
import Modal from '@material-ui/core/Modal';
import { ConnectedRouter } from 'connected-react-router';
import { Provider } from 'react-redux';
import settings from 'electron-settings';
import { History } from 'history';
import { Logger } from 'library/Logger';
import { EVENTS, HEADSET_STATUS } from 'library/headset/consts';
import routes from 'constants/routes';
import TitleBar from 'components/TitleBar';
import { DEFAULT_LANGUAGE_CODE } from 'store/reducers/language';
import { ALL_LANGUAGES } from 'resources/strings/languages';
import styles from './App.css';
import bgImage from '../../resources/images/registration_bg.png';
import exportedStyles from './styles';
import * as Authentication from '../../library/Authentication';
import StyledButton from '../StyledButton';

type Props = {
    store: {},
    history: History,
    classes: {
        [key: string]: string
    }
};

class App extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            signOutConfirmation: false,
            language: this.getLanguage(),
            showSignInScreen:
                !Authentication.isSignedIn() ||
                Authentication.isWaitingForEmailVerification(),
            device: {
                connected: HEADSET_STATUS.HEADSET_DISCONNECTED,
                model: null,
                serial: null,
                fwVersion: null,
                notSupported: false,
                wasInterrupted: false,
                error: false,
                isDfu: false,
                progress: 0
            }
        };
    }

    componentDidMount() {
        Logger.info('App: Mounted');

        // Register for device events
        ipcRenderer.on(
            `${EVENTS.IS_DEVICE_CONNECTED}-push`,
            (event, deviceDetails) => {
                Logger.info(
                    'received IS_DEVICE_CONNECTED-push - ',
                    event,
                    deviceDetails
                );
                this.onDeviceConnection(event, deviceDetails);
            }
        );
        ipcRenderer.on(
            `${EVENTS.IS_DEVICE_CONNECTED}-response`,
            (event, deviceDetails) => {
                Logger.info('received IS_DEVICE_CONNECTED-response');
                this.onDeviceConnection(event, deviceDetails);
            }
        );

        ipcRenderer.on('windowApplicationClose', () => {
            this.onApplicationClose();
        });
    }

    componentWillUnmount() {
        ipcRenderer.removeAllListeners(`${EVENTS.IS_DEVICE_CONNECTED}-push`);
        ipcRenderer.removeAllListeners(
            `${EVENTS.IS_DEVICE_CONNECTED}-response`
        );
        ipcRenderer.removeAllListeners('windowApplicationClose');
    }

    onApplicationClose() {
        const {
            history: { location }
        } = this.props;
        Logger.info('Checking App on closing');
        if (location.pathname !== routes.UPDATE_DEVICE.path) {
            ipcRenderer.send('QUIT_CONFIRM');
        }
    }

    onDeviceConnection(event, deviceDetails) {
        Logger.info(
            'onDeviceConnection: ',
            deviceDetails.connected,
            deviceDetails.progress,
            deviceDetails.isDfu,
            deviceDetails.isOld,
            deviceDetails
        );
        const { device } = this.state;

        if (
            deviceDetails.fw_error !== device.fwError ||
            deviceDetails.connected !== device.connected ||
            deviceDetails.progress !== device.progress ||
            deviceDetails.isDfu !== device.isDfu ||
            deviceDetails.isOld !== device.isOld ||
            deviceDetails.fw_version !== device.fwVersion ||
            deviceDetails.wasInterrupted !== device.wasInterrupted ||
            deviceDetails.btm_version !== device.btm_version
        ) {
            // Update device connection state and device details
            this.setState(state => {
                const newDevice = clone(state.device);
                newDevice.fwError = deviceDetails.fw_error;
                newDevice.connected = deviceDetails.connected;
                newDevice.progress = deviceDetails.progress;
                newDevice.model = deviceDetails.model;
                newDevice.serial = deviceDetails.serial;
                newDevice.fwVersion = deviceDetails.fw_version;
                newDevice.btName = deviceDetails.bt_name;
                newDevice.btmVersion = deviceDetails.btm_version;
                newDevice.notSupported = deviceDetails.not_supported;
                newDevice.wasInterrupted = deviceDetails.wasInterrupted;
                newDevice.error = deviceDetails.error;
                newDevice.isDfu = deviceDetails.isDfu;
                newDevice.isOld = deviceDetails.isOld;
                newDevice.deviceLanguage = deviceDetails.device_language;
                if (deviceDetails.isOld) newDevice.notSupported = true;

                if (
                    newDevice.connected &&
                    newDevice.serial &&
                    newDevice.model &&
                    newDevice.fwVersion
                ) {
                    if (
                        newDevice.connected !== device.connected ||
                        newDevice.serial !== device.serial ||
                        newDevice.model !== device.model ||
                        newDevice.fwVersion !== device.fwVersion
                    ) {
                        if (
                            Authentication.isSignedIn() &&
                            !Authentication.isWaitingForEmailVerification()
                        ) {
                            Authentication.deviceConnected(newDevice);
                        }
                    }
                }

                return {
                    ...state,
                    device: newDevice
                };
            });
        }
    }

    onAuthenticationChange() {
        const { device } = this.state;

        const currentShowSignInScreen =
            !Authentication.isSignedIn() ||
            Authentication.isWaitingForEmailVerification();

        this.setState({
            showSignInScreen: currentShowSignInScreen
        });

        if (!currentShowSignInScreen && device.connected) {
            Authentication.deviceConnected(device);
        }
    }

    renderRoutes() {
        const { language, device } = this.state;
        // Pass the device state, strings (according to language), etc. to each router
        const componentProps = {
            device,
            language,
            onLanguageSet: lang => {
                this.setLanguage(lang);
            },
            onAuthenticationChange: () => {
                this.onAuthenticationChange();
            },
            strings: ALL_LANGUAGES[language].values // Return selected language strings
        };

        return (
            <Switch>
                {Object.values(routes).map(({ path, component }) => (
                    <Route
                        key={path}
                        exact
                        path={path}
                        render={routeProps =>
                            React.createElement(component, {
                                ...routeProps,
                                ...componentProps
                            })
                        }
                    />
                ))}
            </Switch>
        );
    }

    setLanguage(language) {
        settings.setSync('language', language);
        this.setState({ language });
    }

    getLanguage() {
        let stateLanguage;
        if (this.state) {
            stateLanguage = this.state.language; // eslint-disable-line
        }
        let languageSettings = settings.getSync('language');
        if (stateLanguage && stateLanguage !== languageSettings) {
            this.setLanguage(stateLanguage);
            languageSettings = stateLanguage;
        }
        if (!languageSettings) {
            this.setLanguage(DEFAULT_LANGUAGE_CODE);
            languageSettings = DEFAULT_LANGUAGE_CODE;
        }
        return languageSettings;
    }

    moveToScreen(path) {
        const { history } = this.props;
        Logger.info('App: moveToScreen: ', path);

        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            history.replace(path);
        }, 100);
    }

    render() {
        const { store, history, classes } = this.props;
        const { showSignInScreen, signOutConfirmation } = this.state;
        const strings = ALL_LANGUAGES[this.getLanguage()].values;

        return (
            <Provider store={store}>
                <ConnectedRouter history={history}>
                    <div
                        className={styles.container}
                        style={
                            showSignInScreen
                                ? {
                                      backgroundImage: `linear-gradient(rgba(0,0,0,0.3),rgba(0,0,0,0.0)), url(${bgImage})`,
                                      backgroundSize: 'cover'
                                  }
                                : {}
                        }
                    >
                        <TitleBar
                            strings={strings}
                            history={history}
                            language={this.getLanguage()}
                            onLanguageSet={language =>
                                this.setLanguage(language)
                            }
                            onSignOut={() => {
                                this.setState({ signOutConfirmation: true });
                            }}
                        />

                        <div className={styles.content}>
                            {this.renderRoutes()}
                        </div>

                        <Modal open={signOutConfirmation}>
                            <div className={classes.modalContainer}>
                                <div className={classes.modalTitle}>
                                    {strings.sign_out}
                                </div>
                                <div className={classes.modalDescription}>
                                    {strings.sign_out_confirmation}
                                </div>
                                <div className={classes.modalButtonContainer}>
                                    <StyledButton
                                        variant="contained"
                                        btnSize="small"
                                        onClick={() => {
                                            this.setState({
                                                signOutConfirmation: false
                                            });

                                            Authentication.signOut();
                                            this.onAuthenticationChange();
                                            this.moveToScreen(
                                                routes.NOT_LOGGED_IN.path
                                            );
                                        }}
                                    >
                                        {strings.yes}
                                    </StyledButton>
                                    <StyledButton
                                        variant="contained"
                                        btnSize="small"
                                        onClick={() => {
                                            this.setState({
                                                signOutConfirmation: false
                                            });
                                        }}
                                    >
                                        {strings.no}
                                    </StyledButton>
                                </div>
                            </div>
                        </Modal>
                    </div>
                </ConnectedRouter>
            </Provider>
        );
    }
}

export default withStyles(exportedStyles)(App);
